/*
VocaTest is intended as a small and simple tool for testing vocabularies
Copyright (C) 2003  Olof Holmgren

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

/*
File: VocaTest.java
Author: Olof Holmgren - oloho217@student.liu.se
Created: 2003-05-28
Modified: Every now and then, by Olof Holmgren
*/

import java.util.*;
import java.io.*;

/**
 * Class VocaTest is all there is. This is the program.
 *
 * @author Olof Holmgren - oloho217@student.liu.se
 * @version 1.0
 */
public class VocaTest
{
    //////////////////////////////////////////////////
    // Private fields
    /////////////////////////////////////////////////

    private static LinkedList loadedVoc = new LinkedList();
    private static LinkedList loadedRandomVoc;
    private static boolean isVocLoaded = false;
    private static BufferedReader in;
    

    // The helpfile :-)
    private static String helpString =
         "\nAvailable commands: \nload [file]     - Unloads all currently loaded vocfiles and loads a new one \nappend [file]   - Appends a new vocfile to the current test \nsave [file]     - Saves the currently loaded vocfile(s) to a new file \ntest            - Perform a test on the loaded vocfile(s) \ntestrand        - Like test but the words are in random order \ntestbackw       - Perform a \"backward\" test on the loaded vocfile(s) \ntestrandbackw   - Like testbackw but the words are in random order \n$abort          - Abort currently running test \nlist            - List the currently loaded words \nbuildvoc [file] - Create a small example vocfile \nhelp            - Display this help \nquit, exit      - Quit VocaTest \n\n";


    // String for the sample vocfile
    private static String sampleFile =
	"// An example of a vocfile for VocaTest\n// This is how to make comments\n// Comments can only be made at the beginning of a row\n//\n// The file is organised as:\n// word or sentence in own language | word or sentence in foreign language\n// another word in own language | with translation\n//\n// You get the idea, right?\n// As you can see I've decided to use the \"|\" (pipe) sign as delimeter.\n// Don't like it? Well I don't really care... :-)\n//\n// Some words below. English | Swedish.\nhello | hej\nplanet of the apes | apornas planet\npizza | pizza\nchewing gum | tuggummi";


    ////////////////////////////////////////////
    // Public methods
    ////////////////////////////////////////////

    // Do you really need a comment for this?
    public static void main(String[] args)
    {
	System.out.print("\nWelcome to VocaTest 1.0\nAuthor: Olof Holmgren - oloho217@student.liu.se\nLicense: GPL\n");
	dispHelp();
	in = new BufferedReader(new InputStreamReader(System.in));
	parseCommand();
    }


    //////////////////////////////////////////////////
    // Private methods
    //////////////////////////////////////////////////

    // Display the help
    private static void dispHelp()
    {
	System.out.print(helpString);
    }

    // Read a command from Standard in and call executeCommand
    private static void parseCommand()
    {
	String command;
	System.out.print("<VocaTest> ");
	try
	    {
		command = in.readLine();

		int stop = command.indexOf(" ", 0);
		if (stop == -1) // We've got a command without arguments
		    {
			stop = command.length(); // Then the whole row is a command
		    }
		executeCommand(command.substring(0, stop), command.substring(stop, command.length()).trim());
	    }
	catch (Exception e)
	    {
		System.out.println("Something went wrong when trying to read your input. Tough luck.");
	    }
    }
	
    // Execute a command
    private static void executeCommand(String command, String file)
    {
	if (command.equalsIgnoreCase("load"))
	    {
		if (file.equals(""))
		    {
			System.out.println("And what file am I supposed to load?");
		    }
		else if (loadFile(file))
		    {
			System.out.println("Loaded file: " + file);
		    }
	    }
	else if (command.equalsIgnoreCase("append"))
	    {
		if (file.equals(""))
		    {
			System.out.println("And what file am I supposed to append?");
		    }
		else if (appendFile(file))
		    {
			System.out.println("Appended file: " + file);
		    }
	    }
	else if (command.equalsIgnoreCase("save"))
	    {
		if (file.equals(""))
		    {
			System.out.println("And what file am I supposed to save to?");
		    }
		else
		    {
			saveVocFile(file);
		    }
	    }
	else if (command.equalsIgnoreCase("test"))
	    {
		if (isVocLoaded)
		    {
			startTest(loadedVoc, false);
		    }
		else
		    {
			System.out.println("You have to load a vocfile first");
		    }
	    }
	else if (command.equalsIgnoreCase("testrand"))
	    {
		if (isVocLoaded)
		    {
			Collections.shuffle(loadedRandomVoc);
			startTest(loadedRandomVoc, false);
		    }
		else
		    {
			System.out.println("You have to load a vocfile first");
		    }
	    }
	else if (command.equalsIgnoreCase("testbackw"))
	    {
		if (isVocLoaded)
		    {
			startTest(loadedVoc, true);
		    }
		else
		    {
			System.out.println("You have to load a vocfile first");
		    }
	    }
	else if (command.equalsIgnoreCase("testrandbackw"))
	    {
		if (isVocLoaded)
		    {
			Collections.shuffle(loadedRandomVoc);
			startTest(loadedRandomVoc, true);
		    }
		else
		    {
			System.out.println("You have to load a vocfile first");
		    }
	    }
	else if (command.equalsIgnoreCase("$abort"))
	    {
		System.out.println("That command can only be performed while running a test");
	    }
	else if (command.equalsIgnoreCase("list"))
	    {
		if (isVocLoaded)
		    {
			listVocs(loadedVoc);
		    }
		else
		    {
			System.out.println("You have to load a vocfile first");
		    }
	    }
	else if (command.equalsIgnoreCase("buildvoc"))
	    {
		if (file.equals(""))
		    {
			System.out.println("And what file am I supposed to save the sample vocfile as?");
		    }
		else
		    {
			buildVoc(file);
		    }		
	    }
	else if (command.equalsIgnoreCase("help"))
	    {
		dispHelp();
	    }
	else if (command.equalsIgnoreCase("quit") || command.equalsIgnoreCase("exit"))
	    {
		System.exit(0);
	    }
	else
	    {
		System.out.println("No such command. Type help to see the available commands");
	    }
	parseCommand();
    }

    // Load a vocfile into loadedVoc (a LinkedList)
    private static boolean loadFile(String file)
    {
	loadedVoc = new LinkedList();
	return appendFile(file);
    }

    // Append a vocfile into loadedVoc (a LinkedList)
    private static boolean appendFile(String file)
    {
	boolean couldLoad = false;
	file = "vocfiles" + System.getProperty("file.separator") + file;
	if (fileExists(file))
	    {
		if (canRead(file))
		    {
			try
			    {
				FileReader inFile = new FileReader(file);
				BufferedReader inFileReader = new BufferedReader(inFile);
				
				String row, word;
				int start, stop;
				LinkedList vocPair;
				while ((row = inFileReader.readLine()) != null)
				    {
					if (!(row.equals("")) && !(row.substring(0,2).equals("//")))  // Comments and blank rows be gone!
					    {
						vocPair = new LinkedList();
						start = 0;
						stop = row.indexOf("|");
						word = row.substring(start, stop);
						word = word.trim();
						vocPair.add(word);
						
						start = stop + 1;
						stop = row.length();
						word = row.substring(start, stop);
						word = word.trim();
						vocPair.add(word);
						
						loadedVoc.add(vocPair);
					    }
				    }
				inFile.close();
				inFileReader.close();
				couldLoad = true;
				isVocLoaded = true;

				loadedRandomVoc = (LinkedList) loadedVoc.clone();
			    }
			catch (Exception e1)
			    {
				System.out.println("Something went wrong when trying to read from file: " + file);
			    }
		    }
		else // File exists but cannot be read
		    {
			System.out.println("File " + file + " exists but could not be read. Do you have permission to read this file?");
		    }
	    }
	else // File does not exist
	    {
		System.out.println(file + " does not exist");
	    }
       
	return couldLoad;
    }

    // Create a sample vocfile
    private static void buildVoc(String file)
    {
	file = "vocfiles" + System.getProperty("file.separator") + file;
	boolean goAhead = true;
	if (fileExists(file))
	    {
		if (canWrite(file))
		    {
			try
			    {
				String choice;
				boolean correctAnswer = false;
				System.out.print("Do you want to owerwrite existing file " + file + "? (yes/no) ");
				
				choice = in.readLine().trim();
				while (!correctAnswer)
				    {
					if (choice.equalsIgnoreCase("yes"))
					    {
						correctAnswer = true;
					    }
					else if (choice.equalsIgnoreCase("no"))
					    {
						goAhead = false;
						correctAnswer = true;
					    }
					else
					    {
						System.out.print("Please answer yes or no: ");
						choice = in.readLine().trim();
					    }
				    }
			    }
			catch (Exception e)
			    {
				System.out.println("Something went wrong when trying to read your input. Tough luck.");
			    }
		    }
		else // File exists but could not be written
		    {
			System.out.println("File " + file + " exists but could not be written to. Do you have permission to write to this file?");
			goAhead = false;
		    }
	    }			

	try
	    {
		if (goAhead)
		    {
			FileWriter outFile = new FileWriter(file);
			outFile.write(sampleFile);
			outFile.close();
			System.out.println("Created sample vocfile: " + file);
		    }
		else
		    {
			System.out.println("Sample vocfile " + file + " NOT created");
		    }
	    }
	catch (Exception e)
	    {
		System.out.println("Something went wrong when trying to create the sample vocfile at " + file);
	    }
    }

    // Save the currently loaded vocabulary
    private static void saveVocFile(String file)
    {
	file = "vocfiles" + System.getProperty("file.separator") + file;
	boolean goAhead = true;
	if (fileExists(file))
	    {
		if (canWrite(file))
		    {
			try
			    {
				String choice;
				boolean correctAnswer = false;
				System.out.print("Do you want to owerwrite existing file " + file + "? (yes/no) ");
				
				choice = in.readLine().trim();
				while (!correctAnswer)
				    {
					if (choice.equalsIgnoreCase("yes"))
					    {
						correctAnswer = true;
					    }
					else if (choice.equalsIgnoreCase("no"))
					    {
						goAhead = false;
						correctAnswer = true;
					    }
					else
					    {
						System.out.print("Please answer yes or no: ");
						choice = in.readLine().trim();
					    }
				    }
			    }
			catch (Exception e)
			    {
				System.out.println("Something went wrong when trying to read your input. Tough luck.");
			    }
		    }
		else // File exists but could not be written
		    {
			System.out.println("File " + file + " exists but could not be written to. Do you have permission to write to this file?");
			goAhead = false;
		    }
	    }
			
	try
	    {
		if (goAhead)
		    {
			ListIterator iter = loadedVoc.listIterator();
			String row, vocs = new String();
			LinkedList vocPair;
			while (iter.hasNext())
			    {
				vocPair = (LinkedList) iter.next();		
				row = ((String) vocPair.getFirst()) + " | " + ((String) vocPair.getLast());
				vocs = vocs + "\n" + row;
			    }
			
			FileWriter outFile = new FileWriter(file);
			outFile.write(vocs);
			outFile.close();
			System.out.println("Created vocfile " + file + " from the loaded vocfile(s)");
		    }
		else
		    {
			System.out.println("File NOT saved");
		    }
	    }
	catch (Exception e)
	    {
		System.out.println("Something went wrong when trying to create a new vocfile at " + file);
	    }
    }


    // Checks if a file exists or not
    private static boolean fileExists(String file)
    {
	File f = new File(file);
	return f.exists();
    }

    // Check if we can read from file foo
    private static boolean canRead(String file)
    {
	File f = new File(file);
	return f.canRead();
    }

    // Check if we can write to file foo
    private static boolean canWrite(String file)
    {
	File f = new File(file);
	return f.canWrite();
    }

    // List the currently loaded words
    private static void listVocs(LinkedList vocList)
    {
	ListIterator iter = vocList.listIterator();
	LinkedList vocPair;
	String row;
	while (iter.hasNext())
	    {
		vocPair = (LinkedList) iter.next();
		row = ((String) vocPair.getFirst()) + " | " + ((String) vocPair.getLast());
		System.out.println(row);
	    }
	System.out.print("\nTotal number of words: " + loadedVoc.size() + "\n\n");
    }

    // Start a test on the curerntly loaded vocfile
    private static void startTest(LinkedList vocList, boolean backWards)
    {
	try
	    {
		ListIterator iter = vocList.listIterator();
		LinkedList vocPair;
		int failedWords = 0;
		String word, input;
		boolean abort = false;
		while (iter.hasNext() && !abort)
		    {
			vocPair = (LinkedList) iter.next();

			if (backWards)
			    {
				System.out.println((String) vocPair.getLast());
				word = (String) vocPair.getFirst();
			    }
			else
			    {
				System.out.println((String) vocPair.getFirst());
				word = (String) vocPair.getLast();
			    }

			input = in.readLine().trim();
			if (input.equals("$abort"))
			    {
				abort = true;
			    }
			else if (input.equalsIgnoreCase(word))
			    {
				System.out.print("<<Correct>>\n\n");
			    }
			else
			    {
				System.out.print("Wrong. Try again:\n");
				input = in.readLine().trim();
				if (input.equals("$abort"))
				    {
					abort = true;
				    }
				else if (input.equalsIgnoreCase(word))
				    {
					System.out.print("<<Correct>>\n\n");
				    }
				else
				    {
					System.out.print("Wrong. The correct answer is:\n" + word + "\n\n");
					failedWords++;
				    }
			    }
		    }
		
		if (abort)
		    {
			System.out.println("Test aborted");
		    }
		else
		    {
			int passedWords = vocList.size() - failedWords;
			System.out.println("Number of failed words: " + failedWords);
			System.out.println("Number of correct words: " + passedWords);
		    }
	    }
	catch (Exception e)
	    {
		System.out.println("Something went wrong when trying to read your input. Tough luck.");
	    }
    }
}
